// ========================================
// TV RÉTRO - SCRIPT COMPLET
// ========================================
// ========================================

// Configuration
const TV_CONFIG = {
    scanlineSpeed: 6,
    flickerSpeed: 150,
    glitchChance: 0.02,
    dataUpdateInterval: 2000,
    scrollUpdateInterval: 100
};

// État TV
let tvState = {
    crtEnabled: true,
    scrollProgress: 0,
    dataPoints: 0,
    currentYear: '2019',
    glitchActive: false
};

// ========================================
// ICÔNES SVG TV
// ========================================
const TV_ICONS = {
    // Icône TV
    tv: `<svg viewBox="0 0 64 64" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round">
        <rect x="4" y="12" width="56" height="40" rx="4"/>
        <line x1="12" y1="52" x2="24" y2="52"/>
        <line x1="40" y1="52" x2="52" y2="52"/>
        <line x1="28" y1="48" x2="36" y2="48"/>
        <circle cx="32" cy="32" r="10" fill="none" stroke-width="2"/>
    </svg>`,
    
    // Icône Graphique
    chart: `<svg viewBox="0 0 64 64" fill="none" stroke="currentColor" stroke-width="3" stroke-linecap="round" stroke-linejoin="round">
        <rect x="8" y="8" width="48" height="48" rx="4"/>
        <polyline points="12,48 24,32 36,40 52,16"/>
        <circle cx="24" cy="32" r="3" fill="currentColor" stroke="none"/>
        <circle cx="36" cy="40" r="3" fill="currentColor" stroke="none"/>
        <circle cx="52" cy="16" r="3" fill="currentColor" stroke="none"/>
    </svg>`,
    
    // Icône Flèche Gauche
    chevronLeft: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <polyline points="15 18 9 12 15 6"/>
    </svg>`,
    
    // Icône Flèche Droite
    chevronRight: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <polyline points="9 18 15 12 9 6"/>
    </svg>`,
    
    // Icône Scroll Haut
    scrollUp: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <polyline points="18 15 12 9 6 15"/>
    </svg>`,
    
    // Icône Scroll Bas
    scrollDown: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <polyline points="6 9 12 15 18 9"/>
    </svg>`,
    
    // Icône Statistique
    stats: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M12 20V10"/>
        <path d="M18 20V4"/>
        <path d="M6 20v-4"/>
    </svg>`,
    
    // Icône Clock
    clock: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <circle cx="12" cy="12" r="10"/>
        <polyline points="12 6 12 12 16 14"/>
    </svg>`,
    
    // Icône Signal
    signal: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M5 12.55a11 11 0 0 1 14.08 0"/>
        <path d="M1.42 9a16 16 0 0 1 21.16 0"/>
        <path d="M8.53 16.11a6 6 0 0 1 6.95 0"/>
        <line x1="12" y1="20" x2="12.01" y2="20"/>
    </svg>`,
    
    // Icône Data
    data: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <ellipse cx="12" cy="5" rx="9" ry="3"/>
        <path d="M21 12c0 1.66-4 3-9 3s-9-1.34-9-3"/>
        <path d="M3 5v14c0 1.66 4 3 9 3s9-1.34 9-3V5"/>
    </svg>`,
    
    // Icône CRT
    crt: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <rect x="2" y="3" width="20" height="14" rx="2"/>
        <line x1="8" y1="21" x2="16" y2="21"/>
        <line x1="12" y1="17" x2="12" y2="21"/>
    </svg>`,
    
    // Icône Variance
    variance: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
        <polyline points="3.27 6.96 12 12.01 20.73 6.96"/>
        <line x1="12" y1="22.08" x2="12" y2="12"/>
    </svg>`,
    
    // Icône Correlation
    correlation: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M21 16V8a2 2 0 0 0-1-1.73l-7-4a2 2 0 0 0-2 0l-7 4A2 2 0 0 0 3 8v8a2 2 0 0 0 1 1.73l7 4a2 2 0 0 0 2 0l7-4A2 2 0 0 0 21 16z"/>
    </svg>`,
    
    // Icône Median
    median: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <line x1="3" y1="12" x2="21" y2="12"/>
        <line x1="12" y1="3" x2="12" y2="21"/>
    </svg>`,
    
    // Icône Ecart
    ecart: `<svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <circle cx="12" cy="12" r="10"/>
        <line x1="22" y1="12" x2="18" y2="12"/>
        <line x1="6" y1="12" x2="2" y2="12"/>
        <line x1="12" y1="6" x2="12" y2="2"/>
        <line x1="12" y1="22" x2="12" y2="18"/>
    </svg>`
};

// ========================================
// INITIALISATION GRAPHIQUES TV
// ========================================
function initTVGraphics() {
    console.log('Initialisation graphiques TV...');
    
    // Créer les éléments TV
    createTVStructure();
    
    // Initialiser les composants
    initTVClock();
    initTVSignal();
    initTVScrollGraphic();
    initTVDataCounter();
    initCRTtoggle();
    initNavigation();
    initMetricsInteraction();
    
    // Démarrer les animations
    startTVAnimations();
    
    console.log('Graphismes TV initialisés');
}

// ========================================
// CRÉATION STRUCTURE TV
// ========================================
function createTVStructure() {
    const container = document.getElementById('tvRetroContainer');
    if (!container) return;
    
    // Structure déjà créée dans index.php
    // Ajouter les éléments dynamiques si nécessaire
}

// ========================================
// HORLOGE TV TEMPS RÉEL
// ========================================
function initTVClock() {
    const clockElement = document.getElementById('tvClock');
    if (!clockElement) return;
    
    function updateClock() {
        const now = new Date();
        const hours = String(now.getHours()).padStart(2, '0');
        const minutes = String(now.getMinutes()).padStart(2, '0');
        const seconds = String(now.getSeconds()).padStart(2, '0');
        clockElement.textContent = `${hours}:${minutes}:${seconds}`;
    }
    
    updateClock();
    setInterval(updateClock, 1000);
}

// ========================================
// INDICATEUR DE SIGNAL TV
// ========================================
function initTVSignal() {
    const signalBars = document.querySelectorAll('.signal-bar');
    if (signalBars.length === 0) return;
    
    // Animation des barres de signal
    function animateSignal() {
        signalBars.forEach((bar, index) => {
            setTimeout(() => {
                bar.style.opacity = (0.5 + Math.random() * 0.5).toString();
            }, index * 200);
        });
    }
    
    animateSignal();
    setInterval(animateSignal, 2000);
}

// ========================================
// BARRE DE DÉFILEMENT GRAPHIQUE TV
// ========================================
function initTVScrollGraphic() {
    const scrollTrack = document.querySelector('.tv-scroll-track');
    const scrollPosition = document.querySelector('.tv-scroll-position');
    const scrollCounter = document.querySelector('.tv-scroll-counter');
    
    if (!scrollTrack || !scrollPosition) return;
    
    function updateScrollGraphic() {
        const scrollTop = window.scrollY;
        const docHeight = document.documentElement.scrollHeight - window.innerHeight;
        const progress = docHeight > 0 ? (scrollTop / docHeight) * 100 : 0;
        
        scrollPosition.style.top = `${Math.min(progress, 90)}%`;
        scrollPosition.style.height = `${Math.max(10, 30 - progress * 0.2)}%`;
        
        if (scrollCounter) {
            scrollCounter.textContent = `${Math.round(progress)}%`;
        }
        
        tvState.scrollProgress = progress;
    }
    
    window.addEventListener('scroll', () => {
        requestAnimationFrame(updateScrollGraphic);
    });
    
    // Initialiser
    updateScrollGraphic();
}

// ========================================
// COMPTEUR DE DONNÉES
// ========================================
function initTVDataCounter() {
    const counterElement = document.getElementById('tvDataCounter');
    if (!counterElement) return;
    
    let count = 0;
    
    function updateCounter() {
        // Compteur simulé basé sur le temps
        count = Math.floor(Date.now() / 1000) % 999;
        counterElement.innerHTML = `DONNÉES <span>${count.toString().padStart(3, '0')}</span>`;
    }
    
    updateCounter();
    setInterval(updateCounter, TV_CONFIG.dataUpdateInterval);
}

// ========================================
// TOGGLE CRT ON/OFF
// ========================================
function initCRTtoggle() {
    const toggleBtn = document.getElementById('tvToggleCRT');
    if (!toggleBtn) return;
    
    toggleBtn.addEventListener('click', function() {
        const container = document.getElementById('tvRetroContainer');
        if (!container) return;
        
        tvState.crtEnabled = !tvState.crtEnabled;
        
        if (tvState.crtEnabled) {
            container.classList.remove('crt-disabled');
            this.classList.add('active');
            this.innerHTML = `<span class="led"></span> CRT: ON`;
        } else {
            container.classList.add('crt-disabled');
            this.classList.remove('active');
            this.innerHTML = `<span class="led"></span> CRT: OFF`;
        }
        
        // Effet de transition
        container.classList.add('tv-glitch');
        setTimeout(() => container.classList.remove('tv-glitch'), 300);
    });
}

// ========================================
// NAVIGATION ENTRE GRAPHIQUES
// ========================================
function initNavigation() {
    const leftBtn = document.getElementById('chartLeft');
    const rightBtn = document.getElementById('chartRight');
    
    if (leftBtn) {
        leftBtn.addEventListener('click', function() {
            this.style.transform = 'scale(0.9)';
            setTimeout(() => this.style.transform = '', 150);
            if (typeof navigateChart === 'function') navigateChart(-1);
        });
    }
    
    if (rightBtn) {
        rightBtn.addEventListener('click', function() {
            this.style.transform = 'scale(0.9)';
            setTimeout(() => this.style.transform = '', 150);
            if (typeof navigateChart === 'function') navigateChart(1);
        });
    }
}

// ========================================
// INTERACTION MÉTRIQUES
// ========================================
function initMetricsInteraction() {
    const metrics = document.querySelectorAll('.tv-metric');
    
    metrics.forEach((metric, index) => {
        metric.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
            
            // Mise à jour du compteur de données
            const counter = document.getElementById('tvDataCounter');
            if (counter) {
                const span = counter.querySelector('span');
                if (span) {
                    span.textContent = String(Math.floor(Math.random() * 900 + 100));
                }
            }
        });
        
        metric.addEventListener('mouseleave', function() {
            this.style.transform = '';
        });
        
        metric.addEventListener('click', function() {
            // Effet de sélection
            metrics.forEach(m => m.style.outline = 'none');
            this.style.outline = '2px solid #0066aa';
            this.style.outlineOffset = '2px';
        });
    });
}

// ========================================
// ANIMATIONS TV
// ========================================
function startTVAnimations() {
    // Scanline bar animation
    const scanlineBar = document.querySelector('.tv-scanline-bar');
    if (scanlineBar) {
        scanlineBar.style.animationDuration = '6s';
    }
    
    // Glitch effect occasionnel
    setInterval(() => {
        if (Math.random() < TV_CONFIG.glitchChance && tvState.crtEnabled) {
            triggerGlitchEffect();
        }
    }, 3000);
    
    // Mise à jour position scroll
    setInterval(() => {
        updateScrollIndicators();
    }, TV_CONFIG.scrollUpdateInterval);
}

// ========================================
// EFFET GLITCH
// ========================================
function triggerGlitchEffect() {
    const container = document.getElementById('tvRetroContainer');
    if (!container || !tvState.crtEnabled) return;
    
    tvState.glitchActive = true;
    container.classList.add('tv-glitch');
    
    setTimeout(() => {
        container.classList.remove('tv-glitch');
        tvState.glitchActive = false;
    }, 200);
}

// ========================================
// INDICATEURS DE DÉFILEMENT
// ========================================
function updateScrollIndicators() {
    const scrollTop = window.scrollY;
    const scrollHeight = document.documentElement.scrollHeight;
    const clientHeight = window.innerHeight;
    
    // Mettre à jour la position dans la barre de défilement
    const scrollPosition = document.querySelector('.tv-scroll-position');
    if (scrollPosition && scrollHeight > clientHeight) {
        const percentage = (scrollTop / (scrollHeight - clientHeight)) * 100;
        scrollPosition.style.top = `${Math.min(percentage, 90)}%`;
    }
    
    // Afficher/masquer les flèches de navigation
    const scrollIcons = document.querySelectorAll('.tv-scroll-icon');
    scrollIcons.forEach(icon => {
        if (scrollTop > 50) {
            icon.classList.add('up');
        } else {
            icon.classList.remove('up');
        }
        
        if (scrollTop < scrollHeight - clientHeight - 50) {
            icon.classList.add('down');
        } else {
            icon.classList.remove('down');
        }
    });
}

// ========================================
// NAVIGATION CLAVIER
// ========================================
function initKeyboardNavigation() {
    document.addEventListener('keydown', function(e) {
        switch(e.key) {
            case 'ArrowLeft':
                const leftBtn = document.getElementById('chartLeft');
                if (leftBtn) leftBtn.click();
                break;
            case 'ArrowRight':
                const rightBtn = document.getElementById('chartRight');
                if (rightBtn) rightBtn.click();
                break;
            case 'c':
            case 'C':
                const toggleBtn = document.getElementById('tvToggleCRT');
                if (toggleBtn) toggleBtn.click();
                break;
            case 'ArrowUp':
                window.scrollBy(0, -50);
                break;
            case 'ArrowDown':
                window.scrollBy(0, 50);
                break;
        }
    });
}

// ========================================
// FONCTIONS UTILITAIRES
// ========================================

// Obtenir l'icône SVG
function getTVIcon(iconName) {
    return TV_ICONS[iconName] || '';
}

// Appliquer une icône à un élément
function setTVIcon(elementId, iconName) {
    const element = document.getElementById(elementId);
    if (element) {
        element.innerHTML = getTVIcon(iconName);
    }
}

// Créer une icône SVG inline
function createSVGIcon(iconName, className = '') {
    const icon = getTVIcon(iconName);
    if (!icon) return '';
    
    // Ajouter une classe si spécifiée
    if (className) {
        return icon.replace('<svg ', `<svg class="${className}" `);
    }
    return icon;
}

// ========================================
// EXPORT DES FONCTIONS
// ========================================
window.TV_ICONS = TV_ICONS;
window.initTVGraphics = initTVGraphics;
window.getTVIcon = getTVIcon;
window.setTVIcon = setTVIcon;
window.createSVGIcon = createSVGIcon;
window.triggerGlitchEffect = triggerGlitchEffect;

// ========================================
// INITIALISATION AU CHARGEMENT
// ========================================
document.addEventListener('DOMContentLoaded', function() {
    console.log('TV Graphics: DOM chargé');
    
    // Initialiser avec un léger délai pour s'assurer que tout est prêt
    setTimeout(() => {
        initTVGraphics();
        initKeyboardNavigation();
    }, 100);
});

